from parallels.common.actions.base.subscription_backup_action import SubscriptionBackupAction
from parallels.common.connections_config import is_transfer_resource_limits_enabled
from parallels.common.actions.utils.logging_properties \
	import LoggingProperties
from parallels.common.target_panels import TargetPanels

class RemoveSubscriptionLimitsAndPermissions(SubscriptionBackupAction):
	"""Remove limits and permissions of all subscriptions from Plesk backup file if necessary.
	 1) For target PPA we always remove Plesk limits and permissions from Plesk backup file before restoration. 
		But there are 2 cases:
		1. We don't need to transfer limits, but rather use limits from POA service template.
		   That is a default option for PPA customers.
		2. We need to transfer limits from H-Sphere to PPA for correct billing migration, 
		   we change POA resource limits instead of Plesk limits in function _transfer_resource_limits.
	 2) For target Plesk we have 2 options:
		1. Transfer resource limits from source panel; 
		   in that case we leave limits and permissions in backup file, so they are restored by pleskrestore.
		2. Do not transfer resource limits from source panel, but use resource limits 
		   from target Plesk service plan. In that case we remove limits and permissions from Plesk backup.
	"""

	def get_description(self):
		return "Remove limits and permissions from backup"

	def get_failure_message(self, global_context, subscription):
		return "Failed to remove limits and permissions for subscription '%s'" % (subscription.name)

	def get_logging_properties(self):
		return LoggingProperties(info_log=False)

	def _run_subscription_backup(
		self, global_context, subscription, subscription_backup
	):
		transfer_plesk_resource_limits_enabled = (
			global_context.target_panel == TargetPanels.PLESK
			and 
			is_transfer_resource_limits_enabled(
				global_context.config, 
				global_context.target_panel_obj
			)
		)

		if not transfer_plesk_resource_limits_enabled:
			subscription_backup.remove_limit_and_permissions()
