from contextlib import contextmanager
import ntpath
import posixpath

from parallels.utils import cached
from parallels.common.utils import plesk_utils
from parallels.common.utils import unix_utils
from parallels.common.utils import migrator_utils
from parallels.common.utils import windows_utils


class PleskServer(object):
	@contextmanager
	def runner(self):
		raise NotImplementedError()

	def description(self):
		raise NotImplementedError()

	def is_windows(self):
		raise NotImplementedError()

	@property
	@cached
	def vhosts_dir(self):
		if self.is_windows():
			get_vhosts_dir_function = plesk_utils.get_windows_vhosts_dir
		else:
			get_vhosts_dir_function = plesk_utils.get_unix_vhosts_dir

		with self.runner() as runner:
			return get_vhosts_dir_function(runner)

	def get_vhost_dir(self, domain):
		"""Get path to virtual host's directory of domain"""
		vhost_name = domain.encode('idna')
		if self.is_windows():
			return windows_utils.path_join(self.vhosts_dir, vhost_name)
		else:
			return posixpath.join(self.vhosts_dir, vhost_name)

	@property
	@cached
	def plesk_dir(self):
		if self.is_windows():
			with self.runner() as runner:
				return plesk_utils.get_windows_plesk_dir(runner)
		else:
			with self.runner() as runner:
				return plesk_utils.get_unix_product_root_dir(runner)

	@property
	@cached
	def data_dir(self):
		if self.is_windows():
			with self.runner() as runner:
				return plesk_utils.get_windows_data_dir(runner)
		else:
			with self.runner() as runner:
				#  for Plesk for Unix data dir is the same as root dir
				return plesk_utils.get_unix_product_root_dir(runner)

	@property
	@cached
	def mail_dir(self):
		if self.is_windows():
			raise NotImplementedError()
		else:
			with self.runner() as runner:
				return plesk_utils.get_unix_mailnames_dir(runner)

	@property
	@cached
	def dump_dir(self):
		if self.is_windows():
			with self.runner() as runner:
				dump_dir = plesk_utils.get_windows_dump_dir(runner)
				if not dump_dir:
					# Fallback to %plesk_dir%\Backup
					dump_dir = ntpath.join(self.plesk_dir, u'Backup')
				return dump_dir
		else:
			with self.runner() as runner:
				return plesk_utils.get_unix_dump_dir(runner)

	@cached
	def get_plesk_version(self):
		"""Return a tuple with Plesk server version.
		
		In PPA, Plesk version the same across all service nodes; we get Plesk
		version from management node.
		"""
		if self.is_windows():
			raise NotImplementedError()
		else:
			with self.conn.main_node_runner() as runner:
				return plesk_utils.get_plesk_version(runner)

	@property
	@cached
	def is_centos(self):
		with self.runner() as runner:
			return unix_utils.is_centos(runner)

	@property
	@cached
	def is_debian(self):
		with self.runner() as runner:
			return unix_utils.is_debian(runner)

	@property
	@cached
	def rsync_bin(self):
		with self.runner() as runner:
			return migrator_utils.detect_rsync_path(
				runner, self.description()
			)

	@property
	@cached
	def is_mailserver_qmail(self):
		if self.is_windows():
			return False
		else:
			with self.runner() as runner:
				return 'qmail' in runner.sh(
					'plesk sbin mailmng --features | grep SMTP_Server'
				).lower()

	def get_bin_util_path(self, name):
		if self.is_windows():
			return u'%s\\bin\\%s' % (self.plesk_dir, name)
		else:
			return u'%s/bin/%s' % (self.plesk_dir, name)
