from parallels.common import checking
from parallels.common.plesk_backup.plesk_backup_xml import SubscriptionNotFoundException
from parallels.common.utils import plesk_api_utils
from parallels.utils import find_only, if_not_none


class MigratedSubscription(object):
	"""Objects that represents subscription that we are migrating"""

	def __init__(self, migrator, name):
		self._migrator = migrator
		self._name = name

	@property
	def name_idn(self):
		return self._name.encode('idna')

	@property
	def name(self):
		return self._name

	@property
	def is_windows(self):
		"""Whether the subscription is Windows or Unix"""
		# implementation is not straightforward because of H-Sphere:
		# we can have both Windows and Unix subscriptions in backup
		return self._migrator._subscription_is_windows(
			self._name, self.model.source
		)

	@property
	def is_fake(self):
		"""Check if domain is fake - created by technical reasons

		Fake domains may be not existent on source server, and so they should
		not checked.
		"""
		return self._migrator._is_fake_domain(self._name)

	# ====================== Subscription backups =============================

	@property
	def full_raw_backup(self):
		source_settings = self._migrator.source_plesks[self.model.source]
		return self._migrator.load_raw_plesk_backup(source_settings)

	@property
	def raw_backup(self):
		"""Return subscription from raw backup
		
		Raw backup contains subscriptions like they are
		presented on source panel.

		Returns common.plesk_backup.data_model.Subscription*
		"""
		return self.full_raw_backup.get_subscription(self._name)

	@property
	def raw_mail_backup(self):
		mail_server_id = self._migrator._get_mail_server_id(self.model.source)
		source_settings = self._migrator._get_source_servers()[mail_server_id]
		backup = self._migrator.load_raw_plesk_backup(source_settings)
		try:
			return backup.get_subscription(self._name)
		except SubscriptionNotFoundException:
			# subscription is not presented on centralized mail server
			# so we consider it has no mail
			return None

	@property
	def full_converted_backup(self):
		source_settings = self._migrator.source_plesks[self.model.source]
		return self._migrator.load_converted_plesk_backup(source_settings)

	@property
	def converted_backup(self):
		"""Return subscription from converted backup

		Converted backup contains subscription how 
		we are going to restore them on the target panel.
		So IP addresses are changed, unsupported features are removed,
		etc.

		Returns common.plesk_backup.data_model.Subscription*
		"""
		return self.full_converted_backup.get_subscription(self._name)

	@property
	def converted_mail_backup(self):
		mail_server_id = self._migrator._get_mail_server_id(self.model.source)
		source_settings = self._migrator._get_source_servers()[mail_server_id]
		backup = self._migrator.load_converted_plesk_backup(source_settings)
		try:
			return backup.get_subscription(self._name)
		except SubscriptionNotFoundException:
			# subscription is not presented on centralized mail server
			# so we consider it has no mail
			return None

	# ====================== Target data models ===============================

	@property
	def model(self):
		"""Return target data model subscription
		
		Returns common.target_data_model.Subscription
		"""
		ppa_model = self._migrator._load_ppa_model(False)

		return find_only(
			ppa_model.iter_all_subscriptions(),
			lambda s: s.name == self._name,
			"Failed to find subscription by name"
		)

	@property
	def model_client(self):
		ppa_model = self._migrator._load_ppa_model(False)

		return find_only(
			ppa_model.iter_all_clients(),
			lambda c: self._name in {s.name for s in c.subscriptions},
			"Failed to find client that owns subscription '%s'" % self._name
		)

	@property
	def model_reseller(self):
		ppa_model = self._migrator._load_ppa_model(False)
		for reseller in ppa_model.resellers.itervalues():
			if self.model_client in reseller.clients:
				return reseller

		return None

	# ====================== IP addresses =====================================

	@property
	def target_mail_ip(self):
		"""Return target mail server IP"""
		subscription_target_services = \
			self._migrator._get_subscription_target_services_cached()
		return if_not_none(
			if_not_none(
				subscription_target_services.get(self._name),
				lambda addresses: addresses.mail_ips
			),
			lambda mail_ips: mail_ips.v4
		)

	@property
	def target_web_ip(self):
		"""Return target web server IP"""
		subscription_target_services = \
			self._migrator._get_subscription_target_services_cached()
		return if_not_none(
			if_not_none(
				subscription_target_services.get(self._name),
				lambda addresses: addresses.web_ips
			),
			lambda web_ips: web_ips.v4
		)

	@property
	def source_web_ip(self):
		return self._migrator._get_subscription_content_ip(
			self.model
		)

	@property
	def source_mail_ip(self):
		"""Return source mail server IP"""
		mail_server_id = self._migrator._get_mail_server_id(self.model.source)
		source_settings = self._migrator._get_source_servers()[mail_server_id]
		return self._migrator._get_mailserver_ip_by_subscription_name(
			source_settings, self._name
		).v4

	@property
	def source_mail_ipv6(self):
		mail_server_id = self._migrator._get_mail_server_id(self.model.source)
		source_settings = self._migrator._get_source_servers()[mail_server_id]
		return self._migrator._get_mailserver_ip_by_subscription_name(
			source_settings, self._name
		).v6

	@property
	def target_public_mail_ipv4(self_):
		return self_._get_public_ip(self_.target_mail_ip)

	@property
	def target_public_mail_ipv6(self_):
		return self_._get_public_ip(self_.target_mail_ipv6)

	@property
	def target_public_web_ipv4(self_):
		return self_._get_public_ip(self_.target_web_ip)

	@property
	def target_public_web_ipv6(self_):
		return self_._get_public_ip(self_.target_web_ipv6)

	@property
	def target_mail_ipv6(self):
		subscription_target_services = \
			self._migrator._get_subscription_target_services_cached()
		return if_not_none(
			if_not_none(
				subscription_target_services.get(self._name),
				lambda addresses: addresses.mail_ips
			),
			lambda mail_ips: mail_ips.v6
		)

	@property
	def target_web_ipv6(self):
		subscription_target_services = \
			self._migrator._get_subscription_target_services_cached()
		return if_not_none(
			if_not_none(
				subscription_target_services.get(self._name),
				lambda addresses: addresses.web_ips
			),
			lambda web_ips: web_ips.v6
		)

	@property
	def target_dns_ips(self):
		"""Return IP addresses of target DNS servers"""
		return self._migrator._get_target_panel_api().get_domain_dns_server_ips(self._name)

	@property
	def source_dns_ips(self):
		"""Return IP addresses of source DNS servers"""
		return self._migrator._get_source_dns_ips(self.model.source)

	@property
	def web_target_server(self):
		return self._migrator._get_subscription_nodes(self._name).web

	# ====================== Server objects ===================================

	@property
	def web_source_server(self):
		return self._migrator._get_source_web_node(self._name)

	@property
	def mail_target_server(self):
		return self._migrator._get_subscription_nodes(self._name).mail

	@property
	def mail_source_server(self):
		return self._migrator._get_source_mail_node(self._name)

	@property
	def db_target_servers(self):
		return self._migrator._get_subscription_nodes(self._name).database

	@property
	def dns_target_servers(self):
		return self._migrator._get_subscription_nodes(self._name).dns

	# =============== Subscription status (suspended or active) ===============

	@property
	def suspended_source(self):
		source_server = self.web_source_server
		return self._migrator._is_subscription_suspended_on_source(
			source_server, self._name
		)

	@property
	def suspended_target(self):
		return plesk_api_utils.is_subscription_suspended(
			self._migrator.conn.target.plesk_api(), self._name
		)

	# =============== Additional auxiliary functions ==========================

	def add_report_issue(self, report, problem, solution):
		plain_report = checking.PlainReport(report, *self._migrator._extract_source_objects_info())
		plain_report.add_subscription_issue(
			self.model.source, self._name, problem, solution
		)

	def _get_public_ip(self, ip):
		return {
			ip.ip_address: ip.public_ip_address or ip.ip_address
			for ip in self._migrator._get_target_panel_api().get_all_ips()
		}.get(ip)
