import logging
import posixpath
import re

from parallels.common.actions.base.subscription_action import SubscriptionAction
from parallels.utils import replace_str_prefix
from parallels.common.utils import subscription_filter
from parallels.common.utils import plesk_utils

logger = logging.getLogger(__name__)

class FixUnixPermissions(SubscriptionAction):
	"""Fix file ownership for files in domain directory after moving.
	
	If two source panels contain subscriptions with same system user names,
	rename one of users. After content migration, chown directories
	and files for renamed system user.
	"""
	def get_description(self):
		return "Fix permissions for web files"

	def get_failure_message(self, global_context, subscription):
		return "Failed to fix permissions for web files of subscription '%s'" % (subscription.name)

	def filter_subscription(self, global_context, subscription):
		return subscription_filter.unix_with_virtual_hosting(
			subscription
		)

	def run(self, global_context, subscription):
		source_sysuser_name = subscription.raw_backup.get_phosting_sysuser_name()
		sysuser_name = subscription.converted_backup.get_phosting_sysuser_name()
		target_vhost_dir = posixpath.join(
			subscription.web_target_server.vhosts_dir, 
			subscription.name_idn
		) 
		if sysuser_name is not None and source_sysuser_name != sysuser_name:
			logger.debug(u"Fix file ownership for files in domain directory after moving")
			with subscription.web_source_server.runner() as runner_source:
				source_vhost_dir = plesk_utils.get_unix_vhost_dir(runner_source, subscription.name)
				owner_file_string = runner_source.run(
					'/usr/bin/find', [source_vhost_dir, '-printf', '%u:%p\n']
				)
			owner_file_list = filter(None, owner_file_string.split('\n'))
			for line in owner_file_list:
				regex = re.compile("^(.+):(.+)$")
				search_result = regex.findall(line)
				if len(search_result) != 0:
					(owner, file_path) = search_result[0]
					if owner == source_sysuser_name:
						with subscription.web_target_server.runner() as runner_target:
							runner_target.run("/bin/chown", [
								sysuser_name, 
								replace_str_prefix(
									file_path, source_vhost_dir, target_vhost_dir
							)])
