import logging
from collections import defaultdict
from contextlib import closing

from parallels.common import target_data_model as target_model
from parallels.target_panel_plesk.import_api import model as import_api_model 
from parallels.utils import group_by_id, group_value_by_id, find_first
from parallels.common.checking import Report, Problem
from parallels.common.converter.business_objects.common import \
	index_plesk_backup_domain_objects, \
	check_domain_conflicts, check_subscription_conflicts, check_subscription_already_exists, \
	check_client_contacts_matching_source_panels, \
	check_client_contacts_matching_source_target, \
	index_target_site_names, \
	EntityConverter

class PleskConverter(object):
	"""Generate model of subscriptions and customers 
	ready to import to target panel.
	
	Converter takes the following information:
	- Plesk backup or other source of information 
	about resellers, clients, subscriptions and domains 
	on source servers
	- Information about objects that already exist on
	the target panel: resellers, plans, clients, domains
	and so on
	- Migration list, which contains list of customers and 
	subscriptions to migrate, their mapping between each other,
	plans and resellers

	Converter:
	- Converts each customer and subscription we need to migrate to 
	format ready to import to target panel (clients and subscriptions
	above hosting settings).
	- Performs conflict resolution, for example if client or subscriptions 
	exists on multiple source panel, or already exists on target panel.
	- Performs assignment of subscriptions to clients and plans,
	clients to resellers according to migration list
	- Performs some basic pre-migration checks, for example that plan
	already exists on target panel.

	Result is a model ready to import to target panel.

	Converter is NOT responsible for hosting settings below subscriptions,
	like subdomains, or PHP settings.
	"""

	logger = logging.getLogger(u'%s.Converter' % __name__)

	def __init__(self, admin_user_password, ppa_existing_objects, options, multiple_webspaces=False, entity_converter=None):
		self.target_resellers = {}
		self.target_clients = {}
		self.target_site_names = {}
		self.existing_objects = ppa_existing_objects # XXX rename to target_existing_objects
		if entity_converter is not None:
			self.entity_converter = entity_converter
		else:
			self.entity_converter = EntityConverter(self.existing_objects)

		self.raw_target_clients = {
			None: self._create_fake_client(None) # admin fake client
		}	# these special clients are included here, but not distinguished from other - regular - clients
		self.raw_target_subscriptions = {}

	def convert_plesks(
		self, plesks, plain_report, 
		subscriptions_mapping, customers_mapping, 
		converted_resellers, 
		password_holder, mail_servers=dict(), custom_subscriptions_allowed=False
	):
		self.existing_site_names = index_target_site_names(self.existing_objects)
		self.target_resellers = group_by_id(converted_resellers, lambda r: r.login)

		for plesk_info in plesks:
			self.logger.info(u"Convert backup")
			with closing(plesk_info.load_raw_backup()) as backup:
				self.convert_plesk(
					plesk_info.id, backup, plesk_info.settings.ip, 
					plain_report, 
					plesk_info.settings.is_windows,
					subscriptions_mapping, customers_mapping,
					password_holder,
			)

	def get_ppa_model(self): # XXX rename to get_target_model
		return target_model.Model(plans={}, resellers=self.target_resellers, clients=self.target_clients)

	def fix_emails_of_aux_users(self):
		pass # XXX function applicable for PPA converter only

	def convert_plesk(
		self, plesk_id, backup, ip, plain_report, is_windows, 
		subscriptions_mapping, customers_mapping, 
		password_holder
	):
		self.logger.info(u"Form up target model objects based on source backups")

		# 1. create model objects (clients and subscriptions), but do not add them to model yet
		for subscription in backup.iter_all_subscriptions():
			self._add_subscription(subscription, plesk_id, is_windows, plain_report, backup)

		for client in backup.iter_all_clients():
			if client.login in customers_mapping:
				self._add_client(client, plesk_id, plain_report, password_holder)
		for reseller in self.target_resellers:
			self.raw_target_clients[reseller] = self._create_fake_client(reseller)

		# 2. link model objects
		# 2.1 plans <- subscription, according to migration list

		# dict(owner => set(plan names)), where owner is reseller login or None in case of admin
		plan_names_by_owner = self._get_plan_names_by_owners(self.existing_objects.resellers, self.existing_objects.service_templates)

		for subscription in self.raw_target_subscriptions.values():
			if subscription.source == plesk_id:
				if subscription.name in subscriptions_mapping:
					plan_name = subscriptions_mapping[subscription.name].plan
					owner_login = subscriptions_mapping[subscription.name].owner
					if owner_login is not None:
						if owner_login in customers_mapping:
							# owner is a customer
							reseller_login = customers_mapping.get(owner_login)
						else:
							reseller_login = owner_login # owner is reseller
					else:
						reseller_login = None # owner is administrator

					if plan_name is not None:
						subscription_report = plain_report.get_subscription_report(plesk_id, subscription.name)
						self._check_plan_exists(
							plan_names_by_owner.get(reseller_login, set()), 
							plan_name, reseller_login, subscription_report
						)
						if not subscription_report.has_errors():
							subscription.plan_name = plan_name

							# take addon plans mapping
							if len(subscriptions_mapping[subscription.name].addon_plans) > 0:
								reseller_id = find_first(self.existing_objects.resellers.itervalues(), lambda r: r.contact.username == reseller_login).id if reseller_login is not None else 1
								addon_plans = [st for st in self.existing_objects.addon_service_templates if st.owner_id == reseller_id]
								for addon_plan_name in subscriptions_mapping[subscription.name].addon_plans:
									subscription.addon_plan_ids.append(find_first(addon_plans, lambda p: p.name == addon_plan_name).st_id)
						else:
							# there are critical errors, so do not consider
							# that subscription anymore in further checks and conversions
							del self.raw_target_subscriptions[subscription.name]
					else:
						# custom subscription - subscription that is not assigned to any plan
						# so leave plan_name set to None
						pass 
				else:
					self.logger.debug(u"Subscription '%s' is not listed in migration list, not including it into target model" % subscription.name)
					del self.raw_target_subscriptions[subscription.name]

		# 2.2 subscriptions <- client, according to migration list 
		for subscription in self.raw_target_subscriptions.values():
			if subscription.name in subscriptions_mapping:
				subscription_owner = subscriptions_mapping[subscription.name].owner
				if subscription_owner in self.raw_target_clients:
					# add subscription to model. guard against multiple addition of the same subscription (if it exists on two Plesk servers)
					# TODO think of using different data type for subscriptions of client
					client_subscriptions = self.raw_target_clients[subscription_owner].subscriptions
					client_subcription_names = [ subs.name for subs in client_subscriptions ]
					if subscription.name not in client_subcription_names:
						self.raw_target_clients[subscription_owner].subscriptions.append(subscription)
				else:
					plain_report.add_subscription_issue(
						plesk_id, subscription.name,
						Problem(
							"customer_does_not_exist", Problem.ERROR,
							u"Customer '%s' does not exist in source or in destination panel, subscription will not be transferred." % subscription_owner,
						),
						u"Create the customer with such login manually, or map subscription to any other existing customer."
					)
					del self.raw_target_subscriptions[subscription.name]
			else:
				self.logger.debug(u"Subscription '%s' is not listed in migration list, not including it into target model" % subscription.name)
				del self.raw_target_subscriptions[subscription.name]

		# 2.3 client <- admin, reseller, according to migration list
		def add_client_to_model(client, reseller_login):
			if reseller_login is None:
				if client.login not in self.target_clients:
					self.target_clients[client.login] = client
			elif reseller_login in self.target_resellers:
				reseller_clients_by_login = group_by_id(self.target_resellers[reseller_login].clients, lambda c: c.login)
				if client.login not in reseller_clients_by_login:
					self.target_resellers[reseller_login].clients.append(client)
			else:
				plain_report.add_customer_issue(
					plesk_id, client.login,
					Problem('reseller_does_not_exist', Problem.ERROR,
						u"Client is mapped to an unexisting reseller '%s'" % reseller_login
					),
					u"Create this reseller in destination panel, or map client to some existing reseller"
				)

		reseller_logins = {reseller.login for reseller in backup.iter_resellers()}
		for client in self.raw_target_clients.values():
			if client.login in customers_mapping or client.login is None:
				reseller_login = customers_mapping.get(client.login)
				add_client_to_model(client, reseller_login)
			elif client.login in reseller_logins:
				add_client_to_model(client, client.login)
			else:
				self.logger.debug(u"Client '%s' is not listed in migration list, not including it into target model" % client.login)
				del self.raw_target_clients[client.login]

	def _add_client(self, client, plesk_id, plain_report, password_holder):
		client_report = plain_report.get_customer_report(plesk_id, client.login)
		if client.login in self.existing_objects.customers:
			existing_client = self.existing_objects.customers[client.login]
			new_client = self.entity_converter.create_client_from_plesk_backup_client(
				client, None, Report('', ''), password_holder
			) # just for easy comparison
			check_client_contacts_matching_source_target(new_client, existing_client, client_report)
			target_model_client = self.entity_converter.create_client_stub_from_existing_client(existing_client)
			self.raw_target_clients[client.login] = target_model_client
		elif client.login in self.raw_target_clients:
			existing_client = self.raw_target_clients[client.login]
			new_client = self.entity_converter.create_client_from_plesk_backup_client(
				client, None, Report('', ''), password_holder
			) # just for easy comparison
			check_client_contacts_matching_source_panels(new_client, existing_client, client_report)
			target_model_client = existing_client
		else:
			target_model_client = self.entity_converter.create_client_from_plesk_backup_client(client, plesk_id, client_report, password_holder)
			self.raw_target_clients[client.login] = target_model_client 

		return target_model_client

	def _check_plan_exists(self, reseller_plans, plan_name, reseller_login, subscription_report):
		"""
		Get plan object for specified subscription 
		Return tuple (is_success, plan) where:
		- is_success is a boolean that means if plan selection was successful or not.
		- plan is a plan object or None in case of custom subscription
		"""
		if plan_name not in reseller_plans:
			subscription_report.add_issue(
				Problem(
					"plan_does_not_exist", Problem.ERROR,
					u"%s does not have a service template named '%s', unable to make a subscription on it. Subscription will not be transferred." % (u"reseller '%s'" % reseller_login if reseller_login is not None else "admin", plan_name)	
				),
				u"Create a service template with this name, or assign subscription to other service template"
			)

	@classmethod
	def _get_plan_names_by_owners(cls, target_resellers, target_service_templates):
		"""
		Returns the following structure: plans[owner] = set(plan names)
		where owner is reseller login or None in case of admin.
		"""

		resellers_by_id = group_value_by_id(
			target_resellers.itervalues(), 
			key_func=lambda reseller: reseller.id, 
			value_func=lambda reseller: reseller.contact.username
		)
		resellers_by_id[import_api_model.ADMIN_ACCOUNT_ID] = None # special admin owner

		plans = defaultdict(set)
		for plan in target_service_templates:
			if plan.owner_id in resellers_by_id:
				reseller = resellers_by_id.get(plan.owner_id)
				plans[reseller].add(plan.name)

		return dict(plans)

	def _add_subscription(self, subscription, plesk_id, is_windows, plain_report, backup):
		issues = []

		webspaces_by_name = group_by_id(self.existing_objects.webspaces, lambda ws: ws.name)
		if subscription.name in webspaces_by_name:
			model_subscription = self.entity_converter.create_subscription_stub_from_existing_subscription(
				subscription, plesk_id, is_windows
			)
			issues += check_subscription_already_exists(
				subscription.name, 
				target_webspaces=self.existing_objects.webspaces
			)
		else:
			model_subscription = self.entity_converter.create_subscription_from_plesk_backup_subscription(subscription, plesk_id, is_windows)
			model_subscription.group_name = model_subscription.name
			issues += check_subscription_conflicts(
				subscription.name,
				source_webspaces=self.raw_target_subscriptions,
				source_sites=self.target_site_names,
				target_sites=self.existing_site_names,
			)
			issues += check_domain_conflicts(
				backup, subscription.name,
				target_webspaces=self.existing_objects.webspaces,
				target_sites=self.existing_site_names,
				source_webspaces=self.raw_target_subscriptions,
				source_sites=self.target_site_names,
			)

		for issue in issues:
			plain_report.add_subscription_issue(plesk_id, subscription.name, *issue)

		critical_issues = [ issue for issue in issues if issue[0].severity == Problem.ERROR ]
		if len(critical_issues) == 0:
			self.raw_target_subscriptions[subscription.name] = model_subscription
			domain_objects = index_plesk_backup_domain_objects(backup, subscription.name) 
			for name, kind in domain_objects:
				self.target_site_names[name] = (subscription.name, kind)

	def _create_fake_client(self, login):
		return target_model.Client(
			login=login, password=None, subscriptions=[],
			company=None,
			personal_info=target_model.PersonalInfo(
				first_name=None,
				last_name=None,
				email=None,
				preferred_email_format=None,
				address_line_1=None,
				address_line_2=None,
				city=None,
				county=None,
				state=None, 
				postal_code=None, 
				language_code=None,
				locale=None,
				country_code=None, 
				primary_phone=None, 
				additional_phone=None, 
				fax=None,
				mobile_phone=None,
			),
			auxiliary_user_roles=[],
			auxiliary_users=[],
			is_enabled=True,
			source='ppa',
		)

