"""Module with filters to filter subscriptions we are migrating"""

from parallels.common.migrator_config import MailContent


def copy_mail(subscription):
	"""Filter subscriptions for which we are going to copy mail

	Arguments:
	- subscription - instance of MigrationSubscription
	"""
	source_server = subscription.mail_source_server
	target_server = subscription.mail_target_server

	if source_server is None:
		return False
	if target_server is None:
		return False
	if source_server.mail_settings.mode == MailContent.NONE:
		return False

	return True


def copy_mail_full(subscription):
	"""Filter subscriptions for which we are going to copy mail in full mode

	Full mode usually means usage of imapsync or something else
	which copies all customer's mail messages 

	Arguments:
	- subscription - instance of MigrationSubscription
	"""
	source_server = subscription.mail_source_server
	return (
		copy_mail(subscription) 
		and
		source_server.mail_settings.mode == MailContent.FULL
	)


def copy_mail_messages(subscription):
	"""Filter subscriptions for which we are going to copy mail in messages mode

	Full mode usually means usage of imapsync or something else
	which copies all customer's mail messages 

	Arguments:
	- subscription - instance of MigrationSubscription
	"""
	source_server = subscription.mail_source_server
	return (
		copy_mail(subscription) 
		and
		source_server.mail_settings.mode == MailContent.MESSAGES
	)


def windows_with_virtual_hosting(subscription):
	"""Filter Windows subscriptions which have virtual hosting

	Arguments:
	- subscription - instance of MigrationSubscription
	"""
	return (
		subscription.is_windows 
		and 
		subscription.raw_backup.hosting_type == 'phosting'
		and
		subscription.web_target_server is not None
	)


def unix_with_virtual_hosting(subscription):
	"""Filter Unix subscriptions which have virtual hosting

	Arguments:
	- subscription - instance of MigrationSubscription
	"""
	return (
		not subscription.is_windows 
		and 
		subscription.raw_backup.hosting_type == 'phosting'
		and
		subscription.web_target_server is not None
	)


def windows_with_iis_dedicated_app_pool_enabled(subscription):
	"""Filter Windows subscriptions with dedicated IIS application pool

	Arguments:
	- subscription - instance of MigrationSubscription
	"""
	return (
		windows_with_virtual_hosting(subscription) 
		and
		subscription.raw_backup.dedicated_iis_application_pool_enabled
	)


def windows_with_iis_dedicated_app_pool_disabled(subscription):
	"""Filter Windows subscriptions with shared IIS application pool

	Arguments:
	- subscription - instance of MigrationSubscription
	"""
	return (
		windows_with_virtual_hosting(subscription) 
		and
		not subscription.raw_backup.dedicated_iis_application_pool_enabled
	)
