import logging

from parallels.core.utils.common import is_empty_iterator
from parallels.plesk import messages
from parallels.core.actions.base.subscription_action import SubscriptionAction

logger = logging.getLogger(__name__)


class DeployDatabaseUsers(SubscriptionAction):
    def get_description(self):
        return messages.ACTION_DEPLOY_DATABASE_USERS

    def get_failure_message(self, global_context, subscription):
        return messages.ACTION_DEPLOY_DATABASE_USERS_FAILED.format(subscription_name=subscription.name)

    def is_critical(self):
        return False

    def filter_subscription(self, global_context, subscription):
        return not is_empty_iterator(subscription.converted_dump.iter_all_database_users())

    def run(self, global_context, subscription):
        """Perform creation of database users for given subscription on target Plesk

        :type global_context: parallels.core.global_context.GlobalMigrationContext
        :type subscription: parallels.core.migrated_subscription.MigratedSubscription
        """
        for database_user_dump in subscription.converted_dump.iter_all_database_users():
            if global_context.hosting_repository.database_user.is_exists(database_user_dump.name):
                logger.info(messages.ACTION_DEPLOY_DATABASE_USERS_SKIP_EXISTING.format(
                    database_user_name=database_user_dump.name
                ))
                continue

            is_assimilate = subscription.is_assimilate_database_user(
                database_user_dump.name,
                database_user_dump.dbtype
            )

            if is_assimilate:
                logger.info(messages.ACTION_DEPLOY_DATABASE_USERS_ASSIMILATE_DATABASE_USER.format(
                    database_user_name=database_user_dump.name,
                    subscription_name=subscription.name
                ))
            else:
                logger.info(messages.ACTION_DEPLOY_DATABASE_USERS_CREATE_DATABASE_USER.format(
                    database_user_name=database_user_dump.name,
                    subscription_name=subscription.name
                ))

            try:
                global_context.hosting_repository.database_user.create_from_dump(
                    database_user_dump,
                    subscription.name,
                    is_assimilate
                )
            except Exception:
                logger.debug(messages.LOG_EXCEPTION, exc_info=True)
                # place error into report and proceed to next mail account
                if is_assimilate:
                    error_message = messages.ACTION_DEPLOY_DATABASE_USERS_ASSIMILATE_DATABASE_USER_ERROR
                else:
                    error_message = messages.ACTION_DEPLOY_DATABASE_USERS_CREATE_DATABASE_USER_ERROR
                global_context.safe.fail_subscription(
                    subscription.name,
                    error_message.format(
                        database_user_name=database_user_dump.name,
                        subscription_name=subscription.name
                    ),
                    is_critical=False
                )
                continue
