from parallels.core import messages
import logging
from parallels.core.actions.base.common_action import CommonAction
from parallels.core.utils.common import is_empty, is_empty_iterator
from parallels.core.utils.migrator_utils import format_owner_login

logger = logging.getLogger(__name__)


class DeployPanelUsers(CommonAction):
    def get_description(self):
        return messages.ACTION_RESTORE_AUX_USERS

    def get_failure_message(self, global_context):
        return messages.FAILED_TO_RESTORE_AUX_USERS

    def is_critical(self):
        return False

    def filter_action(self, global_context):
        return not is_empty_iterator(self._iter_panel_users_to_create(global_context))

    def run(self, global_context):
        for panel_user, owner in self._iter_panel_users_to_create(global_context):
            self._create_panel_user(panel_user, owner, global_context)

    def _create_panel_user(self, panel_user, owner, global_context):
        """Create given panel user for given owner in target panel

        :type panel_user: parallels.core.target_data_model.AuxiliaryUser
        :type owner: parallels.core.target_data_model.Client
        :type global_context: parallels.core.global_context.GlobalMigrationContext
        """
        with global_context.safe.try_auxiliary_user(
            format_owner_login(owner.login), panel_user.login,
            messages.ACTION_RESTORE_AUX_USER_CREATE_FAILED.format(panel_user_username=panel_user.login)
        ):
            all_subscriptions = {sub.name for sub in global_context.iter_all_subscriptions()}
            if (
                panel_user.subscription_name is not None and
                panel_user.subscription_name not in all_subscriptions
            ):
                return

            if is_empty(panel_user.password) and not panel_user.is_active:
                # We do not migrate such users, as we can't create auxiliary users without password.
                # We do not report that to customer to avoid flooding report and log with useless messages:
                # impact is very low - you loose useless auxiliary user (which primary goal is logging in panel)
                # which is not able to login to panel.
                logger.debug(messages.ACTION_RESTORE_AUX_USERS_SKIP_EMPTY_PASSWORD.format(
                    panel_user_username=panel_user.login,
                    owner_username=format_owner_login(owner.login)
                ))
                return

            logger.info(messages.ACTION_RESTORE_AUX_USER_CREATE.format(
                panel_user_username=panel_user.login,
                owner_username=format_owner_login(owner.login)
            ))
            global_context.hosting_repository.panel_user.create(panel_user, owner)

    def _iter_panel_users_to_create(self, global_context):
        for owner in global_context.target_model.iter_all_owners():
            for panel_user in owner.auxiliary_users:
                if self._is_skip_panel_user(panel_user, owner, global_context):
                    continue
                yield panel_user, owner

    @staticmethod
    def _is_skip_panel_user(panel_user, owner, global_context):
        if panel_user.login == owner.login:
            return True
        if global_context.hosting_repository.panel_user.is_exists(panel_user.login, owner.login):
            logger.debug(messages.ACTION_RESTORE_AUX_USERS_SKIP_EXISTING.format(
                panel_user_username=panel_user.login,
                owner_username=format_owner_login(owner.login)
            ))
            return True
        return False
