import itertools

from parallels.core import messages
from parallels.core.actions.base.subscription_action import SubscriptionAction
from parallels.core.actions.post_migration_checks.utils import should_run_post_migration_checks
from parallels.core.hosting_check.run_checks import run_entities
from parallels.core.utils.common.logging import create_safe_logger
from parallels.hosting_check import DomainDNSService, DomainDNSForwarding, DomainDNSExternal, DNSRecord

logger = create_safe_logger(__name__)


class TestDnsAction(SubscriptionAction):
    def get_description(self):
        """Get short description of action as string

        :rtype: str | unicode
        """
        return messages.ACTION_TEST_DNS_DESCRIPTION

    def get_failure_message(self, global_context, subscription):
        """
        :type global_context: parallels.core.global_context.GlobalMigrationContext
        :type subscription: parallels.core.migrated_subscription.MigratedSubscription
        :rtype: str | unicode
        """
        return messages.ACTION_TEST_DNS_FAILURE

    def is_critical(self):
        """If action is critical or not

        If action is critical and it failed for a subscription, migration tool
        won't run the next operations for the subscription.

        :rtype: bool
        """
        return False

    def filter_subscription(self, global_context, subscription):
        """Check if we should run this action on given subscription or not

        :type global_context: parallels.core.global_context.GlobalMigrationContext
        :type subscription: parallels.core.migrated_subscription.MigratedSubscription
        :rtype: bool
        """
        return should_run_post_migration_checks(global_context, 'dns')

    def run(self, global_context, subscription):
        """
        :type global_context: parallels.core.global_context.GlobalMigrationContext
        :type subscription: parallels.core.migrated_subscription.MigratedSubscription
        """
        skip_dns_forwarding_test = (
            not global_context.target_panel_obj.is_test_dns_forwarding() or
            not global_context.source_has_dns_forwarding or
            global_context.options.skip_dns_forwarding_test
        )
        skip_dns_external_test = False

        subscription_report = subscription.get_report(global_context.post_migration_check_report_writer)
        converted_subscription = subscription.converted_dump
        for domain in itertools.chain(
            converted_subscription.iter_domains(),
            converted_subscription.iter_aliases()
        ):
            dns_zone_report = subscription_report.subtarget(u'DNS zone', domain.name)
            dns_zone_entities = self._get_check_entities(
                subscription, domain, skip_dns_forwarding_test, skip_dns_external_test
            )
            run_entities(global_context, dns_zone_entities, dns_zone_report)

    @staticmethod
    def _get_check_entities(subscription, domain, skip_dns_forwarding_test, skip_dns_external_test):
        """
        :type subscription: parallels.core.migrated_subscription.MigratedSubscription
        :type domain: parallels.core.dump.data_model.AddonDomain | parallels.core.dump.data_model.Subscription
        :type skip_dns_forwarding_test: bool
        :type skip_dns_external_test: bool
        """
        result = []

        source_dns_ips = subscription.source_dns_ips
        target_dns_ips = subscription.target_dns_ips

        if (
            domain.dns_zone is not None and
            domain.dns_zone.zone_type == 'slave'
        ):
            logger.debug(
                messages.SKIP_CHECKING_DOMAIN_S_BECAUSE_IT, domain.name
            )
            return result

        if (
            domain.dns_zone is None or
            not domain.dns_zone.enabled
        ):
            logger.debug(
                messages.SKIP_CHECKING_DOMAIN_S_BECAUSE_IT_1 % domain.name
            )
            return result

        # DNS zone exists and it is in master mode
        records = [
            DNSRecord(rec_type=r.rec_type, src=r.src, dst=r.dst, opt=r.opt)
            for r in domain.dns_zone.iter_dns_records()
            if (
                # We don't want to check NS records as they could be completely controlled by target panel
                r.rec_type.upper() != 'NS' and
                # We do not set up DNS forwarding for reverse zones as that has
                # no sense. So, skip checking all PTR records.
                r.rec_type.upper() != 'PTR'
            )
            ]

        result.append(
            DomainDNSService(
                domain_name=domain.name,
                dns_records=records,
                dns_servers=target_dns_ips
            )
        )

        if len(target_dns_ips) > 0:
            if not skip_dns_forwarding_test:
                result.append(
                    DomainDNSForwarding(
                        domain_name=domain.name,
                        dns_records=records,
                        source_dns_servers=source_dns_ips,
                        target_dns_server=target_dns_ips[0]
                    )
                )
            if not skip_dns_external_test:
                result.append(
                    DomainDNSExternal(
                        domain_name=domain.name,
                        dns_records=records,
                        source_dns_servers=source_dns_ips,
                        target_dns_servers=target_dns_ips
                    )
                )

        return result
