import os
import shutil
import sys

from parallels.core import messages
from parallels.core.actions.base.common_action import CommonAction
from parallels.core.migration_list.reader.json_reader import MigrationListReaderJSON
from parallels.core.registry import Registry
from parallels.core.utils import unix_utils
from parallels.core.utils import windows_utils
from parallels.core.utils.common import mkdir_p, is_run_on_windows, open_no_inherit
from parallels.core.utils.common.logging import create_safe_logger
from parallels.core.utils.common_constants import LOCK_RUN_QUEUE, LOCK_QUEUE
from parallels.core.utils.entity import Entity
from parallels.core.utils.json_utils import read_json
from parallels.core.utils.locks.file_lock_exception import FileLockException
from parallels.core.utils.locks.session_file_lock import SessionFileLock
from parallels.core.utils.migration_progress import SubscriptionMigrationStatus
from parallels.core.utils.stop_mark import StopMark
from parallels.core.utils.subscription_operations import command_name_to_operation

logger = create_safe_logger(__name__)


class RunQueueAction(CommonAction):
    """Daemon command that takes tasks from the queue and executes them one by one in a separate process

    Only one instance of that command could be running for the same queue.

    For more details on queue tasks you could refer to:
    parallels.core.actions.queue.add_queue_task.AddQueueTaskAction
    """

    def get_description(self):
        """Get short description of action as string

        :rtype: str
        """
        return messages.ACTION_RUN_QUEUE_DESCRIPTION

    def get_failure_message(self, global_context):
        """Get message for situation when action failed

        :type global_context: parallels.core.global_context.GlobalMigrationContext
        """
        return messages.ACTION_RUN_QUEUE_FAILURE

    def run(self, global_context):
        """Run action

        :type global_context: parallels.core.global_context.GlobalMigrationContext
        """
        run_queue_lock = SessionFileLock(LOCK_RUN_QUEUE)
        queue_lock = SessionFileLock(LOCK_QUEUE)

        try:
            # Note: do not release the lock. It will be released automatically once the process is stopped.
            # That is necessary for removing of sessions. When "stop-migration" command acquires that lock,
            # we should be 100% sure that run queue process is terminated.
            # Otherwise there could be issues when removing files, which are still open by run queue process.
            run_queue_lock.acquire()

            tasks_dir = global_context.session_files.get_queue_tasks_dir()
            finished_dir = global_context.session_files.get_finished_tasks_dir()

            queue_lock.acquire_block()
            try:
                mkdir_p(tasks_dir)
                mkdir_p(finished_dir)
            finally:
                queue_lock.release()

            while os.path.isdir(tasks_dir) and not StopMark.is_set():
                queue_lock.acquire_block()
                try:
                    task_ids = os.listdir(tasks_dir)

                    tasks = []
                    for task_id in task_ids:
                        command_file = os.path.join(tasks_dir, task_id, 'command.json')
                        command_info = read_json(command_file)
                        command = command_info['command']
                        tasks.append(TaskInfo(task_id=task_id, command=command))

                    sorted_tasks = sorted(tasks, cmp=compare_tasks)

                    if len(sorted_tasks) > 0:
                        selected_task = sorted_tasks[0]
                        global_context.session_files.get_running_task_file().write(selected_task.task_id)
                    else:
                        selected_task = None
                finally:
                    queue_lock.release()

                if selected_task is not None:
                    # Execute the first task in the queue, then continue to the next iteration
                    # where tasks list will be re-read, as new tasks with more priority may appear.
                    self._run_task(global_context, selected_task.task_id)
                else:
                    # There are no more tasks in the queue - exit

                    # There is possible race condition right here (probability to get it is quite low):
                    # - the first queue runner is running
                    # - a task is added after we checked that there are no tasks in the first runner
                    # - new queue runner is not started as the first one is still running
                    # To resolve that issue, start a new instance of queue runner periodically
                    # (now we do that periodically from GUI on migration status page)

                    return

                queue_lock.acquire_block()
                try:
                    global_context.session_files.get_running_task_file().write("")
                finally:
                    queue_lock.release()
        except FileLockException:
            logger.fwarn(messages.ANOTHER_INSTANCE_OF_RUN_QUEUE_RUNNING)

    @staticmethod
    def _run_task(global_context, task_id):
        """Run single task

        :type global_context: parallels.core.global_context.GlobalMigrationContext
        :type task_id: str | unicode
        :return: None
        """
        tasks_dir = global_context.session_files.get_queue_tasks_dir()
        finished_dir = global_context.session_files.get_finished_tasks_dir()

        logger.finfo(messages.RUN_TASK, task_id=task_id)
        command_file = os.path.join(tasks_dir, task_id, 'command.json')
        if os.path.isfile(command_file):
            with global_context.migrator_server.runner() as runner:
                command_info = read_json(command_file)
                args = [
                    command_info['command'],
                    global_context.session_files.get_path_to_config_file(),
                ]
                migration_list_file = os.path.join(tasks_dir, task_id, 'migration-list.json')
                if os.path.exists(migration_list_file):
                    args.extend([
                        '--migration-list-format=json',
                        '--migration-list-file=%s' % migration_list_file
                    ])
                args.append('--progress-task-id=%s' % task_id)
                args.extend(command_info['args'])

                if is_run_on_windows():
                    command = windows_utils.format_command_list(
                        sys.executable,
                        [
                            sys.argv[0],
                            Registry.get_instance().get_var_dir()
                        ] + args
                    )
                else:
                    command = unix_utils.format_command_list(
                        sys.argv[0], args
                    )

                logger.finfo(messages.RUN_COMMAND, command=command)
                runner.sh_unchecked(command)

                # Mark all subscriptions for which status was not updated as failed
                if os.path.exists(migration_list_file):
                    migration_list_reader = MigrationListReaderJSON(
                        global_context.migrator.get_migration_list_source_data()
                    )
                    operation = command_name_to_operation(command_info['command'])
                    with open_no_inherit(migration_list_file, 'r') as fp:
                        migration_list_data, _ = migration_list_reader.read(fp, subscriptions_only=True)

                    for subscription_name in migration_list_data.subscriptions_mapping.iterkeys():
                        if operation is not None:
                            current_status = global_context.subscriptions_status.get_operation_status(
                                subscription_name, operation
                            )
                            if current_status not in SubscriptionMigrationStatus.finished():
                                global_context.subscriptions_status.set_operation_status(
                                    subscription_name, operation, SubscriptionMigrationStatus.FINISHED_ERRORS
                                )

        logger.finfo(messages.FINISHED_TASK, task_id=task_id)
        shutil.move(os.path.join(tasks_dir, task_id), os.path.join(finished_dir, task_id))


class TaskInfo(Entity):
    def __init__(self, task_id, command):
        self._task_id = task_id
        self._command = command

    @property
    def task_id(self):
        return self._task_id

    @property
    def command(self):
        return self._command


def compare_tasks(task_info_1, task_info_2):
    """Compare 2 tasks to understand order in which they should be executed

    :type task_info_1: parallels.core.actions.queue.run_queue.TaskInfo
    :type task_info_2: parallels.core.actions.queue.run_queue.TaskInfo
    """

    # First, pre-migration check tasks should be executed. Most probably user is waiting for their results
    # to schedule migration. Also these kind of tasks usually work fast and won't make other kinds of tasks
    # to wait a lot.
    if task_info_1.command == 'check' and task_info_2.command != 'check':
        return -1
    if task_info_1.command != 'check' and task_info_2.command == 'check':
        return 1

    # Then compare tasks by date when they were added (task id is a date string) - the earlier the ask was
    if task_info_1.task_id > task_info_2.task_id:
        return 1
    elif task_info_1.task_id < task_info_2.task_id:
        return -1
    else:
        return 0
