from __future__ import absolute_import

import logging
from StringIO import StringIO
from contextlib import closing
from xml.etree import ElementTree

from parallels.core import messages
from parallels.core.utils.clean_logging import clean_xml
from parallels.core.utils.common.logging import BleepingLogger
from parallels.core.utils.config_utils import get_option
from parallels.core.utils.http_client import perform_http_request


class HttpXmlClient(object):
    def __init__(self, url, host=None):
        log_context_length = get_option('log-message-context-length', 0)
        self.logger = BleepingLogger(
            logging.getLogger(__name__), xml_serializer=self._compose_xml,
            context_length=log_context_length)
        self.url = url
        self.host = host
        
    def send_raw(self, data, extra=None):
        """Send raw string to XML API."""
        headers = self._get_http_headers(data, extra)
        return perform_http_request(self.url, data, headers, hostname=self.host)

    def send_xml(self, xml, extra=None):
        """Serialize XML object and send an API request."""
        self.logger.debug(messages.DEBUG_API_REQUEST_TO, self.url, xml)
        request = self._compose_xml(xml)
        response = self.send_raw(request, extra)
        self.logger.debug(messages.DEBUG_API_RESPONSE_FROM, self.url, clean_xml(response))
        return self._parse_xml(response)

    @staticmethod
    def _compose_xml(xml):
        """Serialize ElementTree XML object to string."""
        with closing(StringIO()) as buf:
            xml.write(buf, encoding='utf-8', xml_declaration=True)
            return buf.getvalue()

    @staticmethod
    def _parse_xml(response):
        tree = ElementTree.ElementTree()
        tree.parse(StringIO(response))
        return tree

    def _get_http_headers(self, data, extra=None):
        return {
            'Content-Type': 'text/xml',
            'Pragma': 'no-cache',
        }
