import logging
import json
import textwrap

import parallels.core
from parallels.core import messages
from parallels.core.utils.common import read_unicode_file_contents, format_list, open_no_inherit
from parallels.core.utils.migrator_utils import get_package_extras_file_path

logger = logging.getLogger(__name__)


def write_html_log(text_log_filename, html_log_filename):
    """Convert text log to HTML log with hierarchy tree and other features

    :type text_log_filename: str | unicode
    :type html_log_filename: str | unicode
    """
    errors_count = 0
    warnings_count = 0
    threads = set()

    top_level_item = {
        "message_first_line": "Log start",
        "message_other_lines": [],
        "level": "I",
        "thread": "MT",
        "time": "",
        "context": "context",
        "items": []
    }

    with open_no_inherit(html_log_filename, 'w') as html_fp:
        html_fp.write(read_unicode_file_contents(
            get_package_extras_file_path(parallels.core, 'html_log_header.tpl')
        ))
        html_fp.write(
            'log = {\n'
            '\t"items": [\n'
        )
        with open_no_inherit(text_log_filename) as fp:
            log_item = None

            for line in fp:
                line = line.strip()
                line_parts = line.split('|', 7)
                if len(line_parts) == 8:
                    (
                        continuation, time, level, thread,
                        module, _, context, message
                    ) = line_parts

                    # Mark migration start and end with special level, so you could easily find them in the log
                    if 'MIGRATOR START' in message or 'MIGRATOR END' in message:
                        level = "S"

                    if continuation == "+":
                        if log_item is not None:
                            # Add item to top-level of log items (in case of INFO level or migration start),
                            # or append to another top-level item as a child (in case of any other level)
                            if _is_top_level(log_item):
                                _mark_tree_errors_and_warnings(top_level_item)
                                html_fp.write(_json_log_item(top_level_item) + ",\n")
                                top_level_item = log_item
                            else:
                                top_level_item["items"].append(log_item)

                        wrapped = textwrap.wrap(message, 120)
                        log_item = {
                            "level": level,
                            "message_first_line": wrapped[0] if len(wrapped) > 0 else '',
                            "message_other_lines": wrapped[1:],
                            "thread": thread,
                            "time": time.replace('_', ' '),
                            "context": context,
                            "items": []
                        }

                        # Collect overall log data - number of errors/warnings, threads
                        if log_item["level"] == "E":
                            errors_count += 1
                        elif log_item["level"] == "W":
                            warnings_count += 1
                        threads.add(thread)
                    elif continuation == "=":
                        log_item["message_other_lines"].append(message)
                    else:
                        assert False
                elif len(line_parts) == 2:
                    # Some stack traces are printed as 2-item line, for example:
                    # =|Traceback (most recent call last):
                    # =|  File "C:\Program Files (x86)\Parallels\Plesk ...
                    # ...
                    #
                    # These are always continuation lines of some previous start line (which usually
                    # states that exception occurred).
                    #
                    # Actually that could be fixed on logger side, so we always have 8-item lines,
                    # but that is not an easy task.
                    log_item["message_other_lines"].append(line_parts[1])
                else:
                    logger.warning(messages.INVALID_LOG_LINE.format(
                        log_file=text_log_filename, line=line
                    ))

            if log_item is not None:
                if _is_top_level(log_item):
                    html_fp.write(_json_log_item(top_level_item) + ",\n")
                    top_level_item = log_item
                else:
                    top_level_item["items"].append(log_item)

            _mark_tree_errors_and_warnings(top_level_item)
            html_fp.write(_json_log_item(top_level_item) + "\n")

        html_fp.write(
            '\t],\n'
            '\tthreads: [{threads}],\n'
            '\terrorsCount: {errors_count},\n'
            '\twarningsCount: {warnings_count}\n'
            '}}\n'.format(
                errors_count=errors_count, warnings_count=warnings_count,
                threads=format_list(sorted(threads))
            )
        )
        html_fp.write(read_unicode_file_contents(
            get_package_extras_file_path(parallels.core, 'html_log_footer.tpl')
        ))


def _is_top_level(log_item):
    return log_item["level"] in ("I", "S")


def _json_log_item(item):
    return json.dumps(item, sort_keys=True, indent=4)


def _mark_tree_errors_and_warnings(item):
    has_errors = False
    has_warnings = False
    for child_item in item.get('items', []):
        _mark_tree_errors_and_warnings(child_item)
        if child_item.get('level') == 'W' or child_item.get('has_warnings'):
            has_warnings = True
        if child_item.get('level') == 'E' or child_item.get('has_errors'):
            has_errors = True
    item["has_errors"] = has_errors
    item["has_warnings"] = has_warnings
