from parallels.plesk import messages
from parallels.core.utils.common import find_only
from parallels.plesk.hosting_description.utils import parse_bool_value, safe_get_list, safe_get_struct


class HostingDescriptionModel(object):
    """Object model over hosting description file, which describes hosting on custom panel that should be migrated"""

    def __init__(self, hosting_description_id, data):
        self._hosting_description_id = hosting_description_id
        self._data = data

    def iter_all_subscriptions(self):
        """Iterate over subscriptions to be migrated

        :rtype: collections.Iterable[parallels.plesk.hosting_description.model.HostingDescriptionSubscription]
        """
        for subscription_data in iter_all_subscriptions(self._data):
            yield HostingDescriptionSubscription(subscription_data)

    def get_subscription(self, name):
        """Get subscription by name

        :type name: basestring
        :rtype: parallels.custom_panel_migrator.hosting_description.HostingDescriptionSubscription
        """
        return find_only(
            self.iter_all_subscriptions(), lambda s: s.name == name,
            messages.FAILED_TO_FIND_SUBSCRIPTION_IN_DUMP % name
        )


class HostingDescriptionWebFilesMappingProvider(object):
    """Base class for domain object (webspace, addon domain or subdomain) that provides web files mapping"""
    def __init__(self, data):
        self._data = data

    def iter_web_files(self):
        """Iterate over web file mappings

        :rtype: collections.Iterable[HostingDescriptionWebFilesMapping]
        """
        for web_file_data in safe_get_list(self._data, 'web_files'):
            yield HostingDescriptionWebFilesMapping(web_file_data)


class HostingDescriptionSubscription(HostingDescriptionWebFilesMappingProvider):
    """Object model over subscription in hosting description file"""

    def __init__(self, data):
        super(HostingDescriptionSubscription, self).__init__(data)
        self._data = data

    @property
    def name(self):
        """Name of the subscription

        :rtype: basestring | None
        """
        return self._data.get('name')

    @property
    def source_document_root(self):
        """Description of a path to document root (which is available by HTTP) on the source server

        :rtype: basestring | dict | None
        """
        return self._data.get('source_document_root')

    @property
    def target_document_root(self):
        """Relative path to document root on target server

        Returns None if target panel's default should be used

        :rtype: basestring | None
        """
        return self._data.get('target_document_root')

    @property
    def forwarding_url(self):
        """URL of another site for this site to forward to

        Return None in case of virtual hosting or no hosting.

        :rtype: str | unicode | None
        """
        return self._data.get('forwarding_url')

    @property
    def forwarding_type(self):
        """Type of forwarding: moved permanently, moved temporarily, frame forwarding

        Return None in case of virtual hosting or no hosting.
        Return one of parallels.plesk.hosting_description.model.ForwardingType in case of forwarding.

        :rtype: str | unicode | None
        """
        if 'forwarding_url' in self._data:
            default = ForwardingType.MOVED_PERMANENTLY
        else:
            default = None

        return self._data.get('forwarding_type', default)

    @property
    def no_web_hosting(self):
        """Whether that domain should have no web hosting

        :rtype: bool
        """
        return parse_bool_value(self._data.get('no_web_hosting', False), False)

    @property
    def source_webspace_root(self):
        """Description of a path to webspace root (which contains all web files of webspace) on the source server

        :rtype: basestring | dict | None
        """
        return self._data.get('source_webspace_root')

    @property
    def mail_service(self):
        """Mail service of the subscription

        :rtype: parallels.plesk.hosting_description.model.HostingDescriptionMailService
        """
        data = safe_get_struct(self._data, 'mail_service')
        if data is not None:
            return HostingDescriptionMailService(data)
        else:
            return None

    def iter_databases(self):
        """Iterate over databases of that subscription

        :rtype: collections.Iterable[parallels.custom_panel_migrator.hosting_description.HostingDescriptionDatabase]
        """
        for database_data in safe_get_list(self._data, 'databases'):
            database = HostingDescriptionDatabase(database_data)
            if database.name is not None:
                yield database

    def iter_mailboxes(self):
        """Iterate over mailboxes of that subscription

        :rtype: collections.Iterable[parallels.plesk.hosting_description.model.HostingDescriptionMailbox]
        """
        domains = [self._data] + safe_get_list(self._data, 'addon_domains')
        for domain in domains:
            mail_service = safe_get_struct(domain, 'mail_service')
            if not mail_service:
                continue
            for mail_user_data in safe_get_list(mail_service, 'mail_users'):
                mailbox = HostingDescriptionMailbox(mail_user_data, domain['name'])
                if mailbox.name is not None:
                    yield mailbox

    def iter_addon_domains(self):
        """Iterate over addon domains of that subscription

        :rtype: collections.Iterable[parallels.plesk.hosting_description_model.HostingDescriptionAddonDomain]
        """
        for addon_domain_data in safe_get_list(self._data, 'addon_domains'):
            addon = HostingDescriptionAddonDomain(addon_domain_data)
            if addon.name is not None:
                yield addon

    def iter_subdomains(self):
        """Iterate over subdomains of that subscription

        :rtype: collections.Iterable[parallels.plesk.hosting_description.model.HostingDescriptionSubdomain]
        """
        for subdomain_data in safe_get_list(self._data, 'subdomains'):
            subdomain = HostingDescriptionSubdomain(subdomain_data)
            if subdomain.name is not None:
                yield subdomain

    def get_addon_domain(self, name):
        """Get addon domain by name

        :type name: basestring
        :rtype: parallels.custom_panel_migrator.hosting_description.HostingDescriptionAddonDomain
        """
        return find_only(
            self.iter_addon_domains(), lambda d: d.name == name,
            messages.FAILED_TO_FIND_ADDON_IN_DUMP % name
        )

    def get_subdomain(self, name):
        """Get subdomain by name

        :type name: basestring
        :rtype: parallels.custom_panel_migrator.hosting_description.HostingDescriptionSubdomain
        """
        return find_only(
            self.iter_subdomains(), lambda d: d.name == name,
            messages.FAILED_TO_FIND_SUBDOMAIN_IN_DUMP % name
        )


class HostingDescriptionDatabase(object):
    """Object model over database in hosting description file"""

    def __init__(self, data):
        self._data = data

    @property
    def db_type(self):
        """Type of the database (either 'mysql' or 'mssql')

        :rtype: basestring
        """
        return self._data.get('type', 'mysql')

    @property
    def name(self):
        """Name of the database

        :rtype: basestring
        """
        return self._data.get('name')

    @property
    def dump(self):
        """Path to database dump file

        :rtype: basestring | None
        """
        return self._data.get('dump')

    @property
    def server_id(self):
        """ID of database server from configuration file

        :rtype: str | None
        """
        return self._data.get('server')


class HostingDescriptionMailService(object):
    """Object model over mail service in hosting description file"""

    def __init__(self, data):
        self._data = data

    @property
    def mail_server(self):
        """ID of mail server where mail messages are stored

        :rtype: str | unicode | None
        """
        return self._data.get('mail_server')

    @property
    def mail_ip(self):
        """IPv4 address of mail service of subscription

        :rtype: str | unicode | None
        """
        return self._data.get('mail_ip')

    @property
    def mail_ipv6(self):
        """IPv6 address of mail service of subscription

        :rtype: str | unicode | None
        """
        return self._data.get('mail_ipv6')


class HostingDescriptionMailbox(object):
    """Object model over mailbox in hosting description file"""

    def __init__(self, data, domain_name):
        self._data = data
        self._domain_name = domain_name

    @property
    def name(self):
        """Full name of the mailbox

        :rtype: str
        """
        return '%s@%s' % (self._data['name'], self._domain_name)

    @property
    def short_name(self):
        """Short name of the mailbox - everything before '@' symbol

        :rtype: str
        """
        return self._data['name']

    @property
    def domain_name(self):
        """Name of domain of the mailbox

        :rtype: str
        """
        return self._domain_name

    @property
    def source_directory(self):
        """Directory on the source server that contains mail messages (specific to Unix only)

        :rtype: basestring | None
        """
        return self._data.get('directory')

    @property
    def quota(self):
        """Mailbox quota, as string

        :rtype: str | None
        """
        return self._data.get('disk_quota')


class HostingDescriptionAddonDomain(HostingDescriptionWebFilesMappingProvider):
    """Object model over addon domain in hosting description file"""

    def __init__(self, data):
        super(HostingDescriptionAddonDomain, self).__init__(data)
        self._data = data

    @property
    def name(self):
        """Name of addon domain

        :rtype: str | None
        """
        return self._data.get('name')

    @property
    def source_document_root(self):
        """Description of a path to document root (which is available by HTTP) on the source server

        :rtype: basestring | dict | None
        """
        return self._data.get('source_document_root')

    @property
    def target_document_root(self):
        """Relative path to document root on target server

        Returns None if target panel's default should be used

        :rtype: basestring | None
        """
        return self._data.get('target_document_root')

    @property
    def forwarding_url(self):
        """URL of another site for this site to forward to

        Return None in case of virtual hosting or no hosting.

        :rtype: str | unicode | None
        """
        return self._data.get('forwarding_url')

    @property
    def forwarding_type(self):
        """Type of forwarding: moved permanently, moved temporarily, frame forwarding

        Return None in case of virtual hosting or no hosting.
        Return one of parallels.plesk.hosting_description.model.ForwardingType in case of forwarding.

        :rtype: str | unicode | None
        """
        if 'forwarding_url' in self._data:
            default = ForwardingType.MOVED_PERMANENTLY
        else:
            default = None

        return self._data.get('forwarding_type', default)

    @property
    def no_web_hosting(self):
        """Whether that domain should have no web hosting

        :rtype: bool
        """
        return parse_bool_value(self._data.get('no_web_hosting', False), False)


class HostingDescriptionSubdomain(HostingDescriptionWebFilesMappingProvider):
    """Object model over subdomain in hosting description file"""

    def __init__(self, data):
        super(HostingDescriptionSubdomain, self).__init__(data)
        self._data = data

    @property
    def name(self):
        """Name of the subdomain

        :rtype: str | None
        """
        return self._data.get('name')

    @property
    def parent_domain(self):
        """Name of parent domain, if specified

        :rtype: str | None
        """
        return self._data.get('parent_domain')

    @property
    def source_document_root(self):
        """Description of a path to document root (which is available by HTTP) on the source server

        :rtype: basestring | dict | None
        """
        return self._data.get('source_document_root')

    @property
    def target_document_root(self):
        """Relative path to document root on target server

        Returns None if target panel's default should be used

        :rtype: basestring | None
        """
        return self._data.get('target_document_root')

    @property
    def forwarding_url(self):
        """URL of another site for this site to forward to

        Return None in case of virtual hosting or no hosting.

        :rtype: str | unicode | None
        """
        return self._data.get('forwarding_url')

    @property
    def forwarding_type(self):
        """Type of forwarding: moved permanently, moved temporarily, frame forwarding

        Return None in case of virtual hosting or no hosting.
        Return one of parallels.plesk.hosting_description.model.ForwardingType in case of forwarding.
        :rtype: str | unicode | None
        """
        if 'forwarding_url' in self._data:
            default = ForwardingType.MOVED_PERMANENTLY
        else:
            default = None

        return self._data.get('forwarding_type', default)

    @property
    def no_web_hosting(self):
        """Whether that domain should have no web hosting

        :rtype: bool
        """
        return parse_bool_value(self._data.get('no_web_hosting', False), False)


class HostingDescriptionWebFilesMapping(object):
    """Object model over web files mapping in hosting description file"""

    def __init__(self, data):
        self._data = data

    @property
    def source(self):
        """Source path of the web files to be copied

        :rtype: str | unicode
        """
        return self._data['source']

    @property
    def target(self):
        """Target path of the web files to be copied

        :rtype: str | unicode
        """
        return self._data['target']

    @property
    def exclude(self):
        """List of files to exclude

        :rtype: list[str | unicode]
        """
        return safe_get_list(self._data, 'exclude')

    @property
    def skip_if_source_not_exists(self):
        """Whether to silently skip this item if source path does not exist on the source server

        :rtype: bool
        """
        return self._data.get('skip_if_source_not_exists', False)


class ForwardingType(object):
    MOVED_PERMANENTLY = 'moved-permanently'
    MOVED_TEMPORARILY = 'moved-temporarily'
    FRAME_FORWARDING = 'frame-forwarding'


def iter_all_subscriptions(data):
    """
    :type data: dict
    :rtype: collections.Iterable[dict]
    """
    for reseller_data in safe_get_list(data, 'resellers'):
        for subscription_data in safe_get_list(reseller_data, 'subscriptions'):
            yield subscription_data
        for customer_data in safe_get_list(reseller_data, 'customers'):
            for subscription_data in safe_get_list(customer_data, 'subscriptions'):
                yield subscription_data
    for customer_data in safe_get_list(data, 'customers'):
        for subscription_data in safe_get_list(customer_data, 'subscriptions'):
            yield subscription_data
    for subscription_data in safe_get_list(data, 'subscriptions'):
        yield subscription_data
