import re

from plesk_mail_migrator.utils.hash_utils import HashUtils


class MailMessageUtils(object):
    @classmethod
    def get_normalized_message_id(cls, message_text):
        """Get normalized message ID from message text ("Message-ID" header value)
        for comparison of messages during migration.

        See "How the tool works with message IDs" in README for more details.

        :type message_text: str
        :rtype: str
        """
        match = re.search(r"^message-id:(.*)$", message_text)
        if match:
            return cls.normalize_message_id(match.group(1))
        else:
            return HashUtils.calculate_hash(message_text).lower()

    @staticmethod
    def normalize_message_id(message_id):
        """Normalize message ID taken from message headers ("Message-ID" value)
        for comparison of messages during migration.

        See "How the tool works with message IDs" in README for more details.

        :type message_id: str
        :rtype: str
        """
        if message_id is None:
            return None

        message_id = message_id.strip()
        message_id = message_id.strip('><')

        # Use hash of message ID as message ID, to unify algorithm with MailMigrator for Windows,
        # where this is necessary to avoid issues with long file names.
        message_id = HashUtils.calculate_hash(message_id)

        # Make message ID lowercase, so comparison of message IDs works well in case of source
        # case-insensitive filesystem, which is default case when migrating from Windows.
        message_id = message_id.lower()

        return message_id
