from plesk_mail_migrator.utils.cmd_args.cmd_args_parser import CommandLineArgumentException, CmdArgsParser


class PlainCmdArgsParser(CmdArgsParser):
    """Simple class to parse command-line arguments"""

    def __init__(self, args):
        self._args = args
        self._used_args = set()
        self._args_dict = dict()
        for arg in args:
            if '=' in arg:
                name, value = arg.split('=', 1)
            else:
                name = arg
                value = None

            # Command-line parser does not see difference between, to simplify usage:
            # 1) name=value
            # 2) -name=value
            # 3) --name=value
            if name.startswith('--'):
                name = name[2:]
            elif name.startswith('-'):
                name = name[1:]

            if name in self._args_dict:
                raise CommandLineArgumentException(
                    'Command-line parameter "%s" was specified multiple times' % name
                )

            self._args_dict[name] = value

    def get(self, arg_name):
        if arg_name not in self._args_dict:
            raise CommandLineArgumentException(
                'Missing expected command-line parameter "%s"' % arg_name
            )
        self._used_args.add(arg_name)
        return self._args_dict[arg_name]

    def contains(self, arg_name):
        return arg_name in self._args_dict

    def get_full_param_name(self, short_name):
        return "--%s" % short_name

    def check_unused_arguments(self):
        """Check arguments that were not used. Usually that means that customer specified wrong argument.
        Cheap alternative to explicit validation of arguments set.
        Call that function only once you requested all arguments.

        :rtype: None
        :raises: plesk_mail_migrator.utils.cmd_args.cmd_args_parser.CommandLineArgumentException
        """
        for arg in self._args_dict.iterkeys():
            if arg not in self._used_args:
                raise CommandLineArgumentException(
                    'Unknown command-line parameter "%s"' % arg
                )
