import os
import posixpath
import errno
import messages
from ftplib import FTP, error_perm
from ftp_migrator.tools import safe_format


def log(message):
    print message


class Ftp(object):
    MAX_RETRY = 3

    def __init__(self, host, username, password):
        self._host = host
        self._username = username
        self._password = password
        self._ftp = FTP()
        self.connect()

    def connect(self):
        """Setup connection with remote FTP server

        :return: 
        """
        log(safe_format(messages.CONNECTION, host=self._host, username=self._username))
        self._ftp.connect(self._host)
        self._ftp.login(self._username, self._password)
        log(safe_format(messages.CONNECTION_SUCCESS, host=self._host, username=self._username))

    def close(self):
        """Close connection

        :return: 
        """
        try:
            self._ftp.close()
            log(safe_format(messages.CONNECTION_CLOSED, host=self._host))
        except Exception:
            # There is nothing bad if error occured when closing connection; ignore any error here
            pass

    def download_path(self, remote_path, local_path):
        if self._is_directory(remote_path):
            mkdir_p(local_path)
            # retrieve list of nested items (both files and directories) and process it
            for item in self._list_file_and_directory_paths(remote_path):
                item_basename = posixpath.basename(item)
                if item_basename.startswith('plesk-migrator-agent'):
                    # skip migrator agent files when transferring - they are garbage for the target server
                    # now simply match by name pattern, better implementation should pass list of excluded
                    # directories as argument to the FTP migrator
                    continue
                self.download_path(item, os.path.join(local_path, item_basename))
        else:
            self.download_file(remote_path, local_path)

    def download_file(self, remote_path, local_path):
        log(safe_format(messages.DOWNLOAD_FILE, source_path=remote_path, target_path=local_path))

        def transfer_file():
            try:
                with open(local_path, 'wb') as fp:
                    def _write_target_file_content(content):
                        fp.write(content)
                    self._ftp.retrbinary('RETR %s' % remote_path, _write_target_file_content)
            except error_perm as e:
                log(safe_format(
                    messages.DOWNLOAD_FILE_FAILED, source_path=remote_path,
                    target_path=local_path, message=e.message
                ))

        self._run_ftp_operation(
            transfer_file,
            safe_format(messages.DOWNLOAD_FILE_FAILED, source_path=remote_path, target_path=local_path)
        )

    def _list_file_and_directory_paths(self, path):
        abs_path = posixpath.join('/', path)
        if not self._is_directory(abs_path):
            return []

        log(safe_format(messages.LIST, path=path))
        ftp_list = self._run_ftp_operation(
            lambda: self._ftp.nlst(abs_path), safe_format(messages.LIST_FAILED, path=path)
        )

        paths = []
        for item in ftp_list:
            if item not in ['.', '..']:
                paths.append(posixpath.join(abs_path, item))
        return paths

    def _is_directory(self, path):
        def is_dir():
            try:
                self._ftp.cwd(path)
                self._ftp.cwd('..')
                return True
            except error_perm:
                # change directory command was failed with 5xx error code so consider that given path is a file
                return False

        try:
            return self._run_ftp_operation(
                is_dir, safe_format(messages.CHECK_PATH_FAILED, path=path)
            )
        except Exception:
            # unable to check given path, consider that it is not a directory
            return False

    def _run_ftp_operation(self, ftp_operation_function, error_message):
        error = None
        result = None
        for retry in range(self.MAX_RETRY):
            error = None
            try:
                result = ftp_operation_function()
                if retry > 0:
                    log(safe_format(messages.OPERATION_SUCCESS, retry=retry))
                break
            except Exception as e:
                log(safe_format(messages.OPERATION_FAILED, message=e, retry=retry + 1))
                error = e
                self.close()
                self.connect()
        if error:
            raise Exception(safe_format(error_message, message=error))
        else:
            return result


def mkdir_p(path):
    """Create directory recursive

    :rtype: str | unicode
    :rtype: None
    """
    try:
        os.makedirs(path)
    except OSError as exc:
        if exc.errno == errno.EEXIST and os.path.isdir(path):
            pass
        else:
            raise
