import os
import posixpath
import errno
import ssl
import string
import types

import messages
from ftplib import FTP, error_perm, FTP_TLS
from ftp_migrator.tools import safe_format


def log(message):
    print message


class Ftp(object):
    MAX_RETRY = 3

    def __init__(self, host, username, password, use_ftps):
        """
        :type host: str | unicode
        :type username: str | unicode
        :type password: str | unicode
        :type use_ftps: bool
        """
        self._host = host
        self._username = username
        self._password = password
        self._use_ftps = use_ftps
        self._is_mlsd_supported = None
        self._ftp = None
        self.connect()

    def connect(self):
        """Setup connection with remote FTP server

        :return: 
        """
        log(safe_format(messages.CONNECTION, host=self._host, username=self._username))
        if self._use_ftps:
            self._ftp = SSLSocketUnwrapAwareFtpTls(context=ssl._create_unverified_context())
        else:
            self._ftp = FTP()

        self._ftp.connect(self._host)
        self._ftp.login(self._username, self._password)

        if self._use_ftps:
            self._ftp.prot_p()  # secure the data connection

        log(safe_format(messages.CONNECTION_SUCCESS, host=self._host, username=self._username))

    def close(self):
        """Close connection

        :return: 
        """
        try:
            self._ftp.close()
            log(safe_format(messages.CONNECTION_CLOSED, host=self._host))
        except Exception:
            # There is nothing bad if error occured when closing connection; ignore any error here
            pass

    def download_directory(self, remote_path, local_path):
        abs_remote_path = posixpath.join('/', remote_path)

        mkdir_p(local_path)
        # retrieve list of nested items (both files and directories) and process it
        for item in self._list_file_and_directory_paths(remote_path):
            if item.name.startswith('plesk-migrator-agent'):
                # skip migrator agent files when transferring - they are garbage for the target server
                # now simply match by name pattern, better implementation should pass list of excluded
                # directories as argument to the FTP migrator
                continue

            item_remote_full_path = posixpath.join(abs_remote_path, item.name)
            item_local_full_path = os.path.join(local_path, item.name)

            if item.is_dir:
                self.download_directory(item_remote_full_path, item_local_full_path)
            else:
                self.download_file(item_remote_full_path, item_local_full_path)

    def download_file(self, remote_path, local_path):
        log(safe_format(messages.DOWNLOAD_FILE, source_path=remote_path, target_path=local_path))

        def transfer_file():
            try:
                with open(local_path, 'wb') as fp:
                    def _write_target_file_content(content):
                        fp.write(content)
                    self._ftp.retrbinary('RETR %s' % remote_path, _write_target_file_content)
            except error_perm as e:
                log(safe_format(
                    messages.DOWNLOAD_FILE_FAILED, source_path=remote_path,
                    target_path=local_path, message=e.message
                ))

        self._run_ftp_operation(
            transfer_file,
            safe_format(messages.DOWNLOAD_FILE_FAILED, source_path=remote_path, target_path=local_path)
        )

    def _list_file_and_directory_paths(self, path):
        """
        :type path: str | unicode
        :rtype: list[parallels.core.utils.ftp.ItemInfo]
        """
        log(safe_format(messages.LIST, path=path))

        abs_path = posixpath.join('/', path)

        if not self.is_mlsd_supported():
            def ftp_dir():
                lines = []

                def consume_lines(line):
                    lines.append(line)

                self._ftp.dir(abs_path, consume_lines)

                return lines

            dir_lines = self._run_ftp_operation(
                ftp_dir, safe_format(messages.LIST_FAILED, path=path)
            )

            return self._parse_list_output(dir_lines)
        else:
            def mlsd_dir():
                lines = []

                def consume_lines(line):
                    lines.append(line)

                self._ftp.retrlines('MLSD %s' % abs_path, consume_lines)

                return lines

            mlsd_lines = self._run_ftp_operation(
                mlsd_dir, safe_format(messages.LIST_FAILED, path=path)
            )

            return self._parse_mlsd_output(mlsd_lines)

    def _run_ftp_operation(self, ftp_operation_function, error_message):
        error = None
        result = None
        for retry in range(self.MAX_RETRY):
            error = None
            try:
                result = ftp_operation_function()
                if retry > 0:
                    log(safe_format(messages.OPERATION_SUCCESS, retry=retry))
                break
            except Exception as e:
                log(safe_format(messages.OPERATION_FAILED, message=e, retry=retry + 1))
                error = e
                self.close()
                self.connect()
        if error:
            raise Exception(safe_format(error_message, message=error))
        else:
            return result

    def is_mlsd_supported(self):
        if self._is_mlsd_supported is None:
            try:
                self._is_mlsd_supported = 'mlsd' in self._ftp.sendcmd('FEAT').lower()
            except Exception:
                # in case of any error, consider MLSD command is not supported and fallback to LIST command
                self._is_mlsd_supported = False

            if self._is_mlsd_supported:
                log(messages.FTP_MLSD_ENABLED)

        return self._is_mlsd_supported

    @staticmethod
    def _parse_list_output(lines):
        """Parse output of LIST command, return list of ItemInfo objects.

        The function parses both Windows and Unix formats, format is detected automatically.
        The function is fault tolerant - in case it failed to parse any line,
        it is simply skipped, no exception is raise.

        :type lines: list[str | unicode]
        :rtype: list[parallels.core.utils.ftp.ItemInfo]
        """
        items = []

        for line in lines:
            if line.strip() == '':
                # ignore empty lines
                continue

            if line.startswith('total'):
                # ignore lines like "total 123" stating items count in the directory
                continue

            if line[0] in string.digits:
                # Windows format
                line_parts = line.split(None, 3)

                if len(line_parts) != 4:
                    # invalid line - skip it
                    continue

                name = line_parts[3]

                if line_parts[2].lower() == '<dir>':
                    item_type = ItemInfo.TYPE_DIR
                else:
                    item_type = ItemInfo.TYPE_FILE
            else:
                # Unix format

                # Two formats are possible: 8-item and 9-item. First detect format by checking the 4-th item,
                # which in case of 9-item format should digits, while in case of 8-digit format should contain
                # letters (month name).
                #
                # Example of 9-item format:
                # drwxr-x--- 5 wpabaturin 99 4096 Apr 26 09:36 public_html
                #
                # Example of 8-item format:
                # -rw-r--r-- 1 staff 3015 May 5 16:08 .bash_aliases
                #
                # See http://ftputil.sschwarzer.net/trac/ticket/12 for details on these formats.
                line_parts = line.split(None, 8)

                if len(line_parts) < 7:
                    # invalid line - skip it
                    continue

                if not all(c in string.digits for c in line_parts[4]):
                    line_parts = line.split(None, 7)

                name = line_parts[len(line_parts) - 1]

                file_type = line_parts[0].lower()[:1]

                if file_type == 'd':
                    item_type = ItemInfo.TYPE_DIR
                elif file_type == '-':
                    item_type = ItemInfo.TYPE_FILE
                else:
                    # we are not interested in sockets, character devices, symbolic links, etc
                    continue

            if name in ('.', '..'):
                continue

            items.append(ItemInfo(name, item_type))

        return items

    @staticmethod
    def _parse_mlsd_output(lines):
        """Parse output of MSDL command according to RFC3659, return list of ItemInfo objects.

        :type lines: list[str | unicode] 
        :rtype: list[parallels.core.utils.ftp.ItemInfo]
        """
        items = []

        for line in lines:
            if line.strip() == '':
                # ignore empty lines; MLSD output should not contain such lines, just for better fault tolerance
                continue

            parts = line.split(None, 1)

            if len(parts) != 2:
                # ignore invalid lines; MLSD output should not contain such lines, just for better fault tolerance
                continue

            facts = {}

            facts_str, filename = parts
            facts_str_list = facts_str.split(';')

            for fact_str_parts in facts_str_list:
                fact_str_parts = fact_str_parts.split('=', 1)

                if len(fact_str_parts) != 2:
                    continue

                name, value = fact_str_parts

                facts[name.lower()] = value

            if facts.get('type') == 'dir':
                item_type = ItemInfo.TYPE_DIR
            elif facts.get('type') == 'file':
                item_type = ItemInfo.TYPE_FILE
            else:
                continue

            items.append(ItemInfo(filename, item_type))

        return items


class ItemInfo(object):
    TYPE_FILE = 'file'
    TYPE_DIR = 'dir'

    def __init__(self, name, item_type):
        self._name = name
        self._item_type = item_type

    @property
    def name(self):
        return self._name

    @property
    def item_type(self):
        return self._item_type

    @property
    def is_dir(self):
        return self.item_type == ItemInfo.TYPE_DIR


def mkdir_p(path):
    """Create directory recursive

    :rtype: str | unicode
    :rtype: None
    """
    try:
        os.makedirs(path)
    except OSError as exc:
        if exc.errno == errno.EEXIST and os.path.isdir(path):
            pass
        else:
            raise


class SSLSocketUnwrapAwareFtpTls(FTP_TLS):
    """Class to workaround compatibility between FTP_TLS class and IIS FTP

    FTP_TLS class tries to call "unwrap" method of SSL socket once the socket for data connection
    is not needed anymore. But IIS seems to close connection before we call "unwrap" for SSL shutdown.
    It makes "unwrap" function to fail which leads to failure of every command which uses FTP data channels.

    Actually, it seems that it is not required to call "unwrap" - connection will be closed anyway. So we replace
    unwrap function with empty one for each data channel socket.

    See http://bugs.python.org/issue10808 for additional details.
    """

    def transfercmd(self, *args, **kwargs):
        sock = FTP_TLS.transfercmd(self, *args, **kwargs)
        sock.unwrap = types.MethodType(lambda _: None, sock)
        return sock

