# Copyright 1999-2016. Parallels IP Holdings GmbH. All Rights Reserved.
package MySQLAgent::Backend::DBI;

use strict;
use MySQLAgent::Backend::Base;
use MySQLAgent::Backend::Utils;

use vars qw|@ISA|;

@ISA = qw|MySQLAgent::Backend::Base|;

# ============================ Public methods =================================

sub connect {
  my ($self) = @_;

  foreach my $requiredModule ('DBI', 'DBD::mysql') {
    if (!_loadModule($requiredModule)) {
      $self->_set_errstr("Unable to find $requiredModule module");
      return;
    }
  }

  my $address = $self->_composeDbiAddress();
  $self->{dbh} = DBI->connect($address, $self->{user}, $self->{password});

  if ($self->{dbh}) {
    if (MySQLAgent::Backend::Utils::doesSupportCharacterSets($self)) {
        $self->_execute(MySQLAgent::Backend::Utils::getCharacterSetsSupportSql());
        $self->_execute(MySQLAgent::Backend::Utils::getNamesCharacterSetsSupportSql());
    }
  } else {
    $self->_set_errstr("Unable to connect to database: $DBI::errstr", $DBI::err);
  }
}

sub disconnect {
  my ($self) = @_;
  $self->{dbh}->disconnect();
}

sub queryArrayOfHashes {
    my ($self, $query, @params) = @_;
    my @result;

    $self->_execute($query, @params);
    while ( my $ptrHash = $self->_fetchHash() ) {
        push @result, $ptrHash;
    }

    $self->_finish();

    return \@result;
}

# =========================== Private methods =================================

sub _execute {
  my ($self, $sql, @params) = @_;

  $self->{result} = $self->{dbh}->prepare($sql);
  if (!$self->{result}) {
    $self->_set_errstr("Unable to prepare SQL ( for execution: $DBI::errstr");
    return;
  }

  my $res = $self->{result}->execute(@params);
  if (!defined $res) {
    $self->_set_errstr("Unable to execute SQL: $DBI::errstr");
    return;
  }

  return $res if $res;
}

sub _fetchHash {
  my ($self) = @_;

  my $h = $self->{result}->fetchrow_hashref();
  return if ref($h) eq "HASH" and !%$h;
  return $h;
}

sub _finish {
  my ($self) = @_;

  $self->{result}->finish();
}

sub _loadModule {
  my ($moduleName) = @_;

  eval "require $moduleName";
  return !$@;
}

sub _composeDbiAddress {
  my ($self) = @_;

  my $address = "DBI:mysql:dbname=$self->{database}";
  $address .= ";host=$self->{host}" if defined $self->{host};
  $address .= ";mysql_socket=$self->{socket}" if defined $self->{socket};
  $address .= ";port=$self->{port}" if defined $self->{port};
  return $address;
}

1;
