<?php
// Copyright 1999-2017. Plesk International GmbH. All rights reserved.

class Dispatcher {
    private $_logger;
    private $_result_encoding;

    // provide results in JSON format
    const RESULT_ENCODING_JSON_PLAIN = 'json_plain';
    // provide results in JSON format, where each string value is encoded with base64,
    // useful to encode binary (non-UTF-8 data)
    const RESULT_ENCODING_JSON_BASE64 = 'json_base64';

    public function __construct() {
        if (!WEB_RPC_AGENT_DEBUG) {
            ob_start();
        }

        $this->_logger = Registry::get_instance()->set_logger(new Logger());
        Registry::get_instance()->set_php_functions(new PhpFunctions());

        if (array_key_exists('result_encoding', $_GET)) {
            if ($_GET['result_encoding'] == self::RESULT_ENCODING_JSON_BASE64) {
                $this->_result_encoding = self::RESULT_ENCODING_JSON_BASE64;
            } else {
                // in case of any invalid value fallback to plain JSON
                $this->_result_encoding = self::RESULT_ENCODING_JSON_PLAIN;
            }
        } else {
            $this->_result_encoding = self::RESULT_ENCODING_JSON_PLAIN;
        }

        register_shutdown_function(array($this, 'handle_shutdown'));
        set_exception_handler(array($this, 'handle_exception'));
        set_error_handler(array($this, 'handle_error'));
    }

    public function run() {
        $task_name = $_GET['task'];

        $rawData = json_decode(file_get_contents("php://input"), true);
        if (!is_array($rawData)) {
            $rawData = array();
        }
        $postData = $_POST;
        if (!is_array($postData)) {
            $postData = array();
        }
        $data = array_merge($rawData, $postData);

        $this->_logger->info('Perform task ' . $task_name);

        $task_name_parts = array();
        foreach (explode('_', $task_name) as $part) {
            $task_name_parts[] = ucfirst($part);
        }
        $task_class_name = implode('', $task_name_parts) . 'Task';
        $task_instance = new $task_class_name($data);

        if (!$task_instance instanceof Task) {
            throw new Exception('Unknown task "' . $task_name . '"');
        }

        try {
            $encoder = $this->_createEncoder();
            $task_instance->run($encoder);
            $this->_finalize($encoder->getData(), null);
        } catch (TaskException $e) {
            $this->_finalize(null, $e->getMessage());
        }
    }

    public function handle_shutdown() {
        $error = error_get_last();
        if (is_null($error)) {
            return;
        }
        switch ($error['type']) {
            case E_ERROR:
            case E_PARSE:
            case E_COMPILE_ERROR:
            case E_CORE_ERROR:
                $error_message = 'PHP Fatal Error at line ' . $error['line'] . ' of file ' . $error['file'] . ': ' .
                    $error['message'];
                $this->_logger->error($error_message);
                $this->_finalize(null, $error_message);
        }
    }

    public function handle_exception(Exception $exception) {
        $error_message = 'Unhandled exception at line ' . $exception->getLine() . ' of file ' .
            $exception->getFile() . ': ' . $exception->getMessage();
        $this->_logger->error($error_message );
        $this->_logger->debug('Exception stacktrace:' . Logger::CRLF . $exception->getTraceAsString());
        $this->_finalize(null, $error_message);
    }

    public function handle_error($errno, $errstr, $errfile, $errline) {
        switch ($errno) {
            case E_WARNING:
                $this->_logger->warning(
                    'PHP Warning at line ' . $errline . ' of file ' . $errfile . ': ' . $errstr
                );
                break;
            case E_NOTICE:
                $this->_logger->debug(
                    'PHP Notice at line ' . $errline . ' of file ' . $errfile . ': ' . $errstr
                );
                break;
            default:
                $this->_logger->debug(
                    'Unknown PHP error ' . $errno . ' at line ' . $errline . ' of file ' . $errfile . ': ' . $errstr
                );
        }
        return true;
    }

    function _finalize($result, $error) {
        if (!WEB_RPC_AGENT_DEBUG) {
            ob_end_clean();
        }

        if (!is_null($error)) {
            http_response_code(500);
        }

        $encoder = $this->_createEncoder();

        $encoder->startDictionary();
        $encoder->startDictionaryValue('result');
        if (count($result) > 0) {
            $encoder->addRawValue($result);
        } else {
            $encoder->addNull();
        }
        $encoder->startDictionaryValue('error');
        $encoder->addPhpValue($error);
        $encoder->endDictionary();

        echo $encoder->getData();

        exit();
    }

    /**
     * @return ResultEncoder
     */
    private function _createEncoder()
    {
        if ($this->_result_encoding == self::RESULT_ENCODING_JSON_BASE64) {
            return new ResultEncoderJsonBase64();
        } else {
            return new ResultEncoderJsonPlain();
        }
    }
}